/* Copyright (C) 2012, Stefan Hacker <dd0t@users.sourceforge.net>

   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   - Redistributions of source code must retain the above copyright notice,
     this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright notice,
     this list of conditions and the following disclaimer in the documentation
     and/or other materials provided with the distribution.
   - Neither the name of the Mumble Developers nor the names of its
     contributors may be used to endorse or promote products derived from this
     software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef MUMBLE_SIGNAL_CURRY_H_
#define MUMBLE_SIGNAL_CURRY_H_

#include <QObject>
#include <QVariant>

/**
 * \brief Small class for binding a QVariant parameter to a parameterless signal.
 * Static usage:
 *    SignalCurry::curry(sender, SIGNAL(emitted()), receiver, SLOT(called(QVariant)), curryParameter);
 */
class SignalCurry : public QObject {
	Q_OBJECT
	Q_DISABLE_COPY(SignalCurry)
private:
	bool bDeleteAfterFirstUse;
	QVariant qvData;
public:
	SignalCurry(QVariant data, bool deleteAfterFirstUse = false, QObject *p = 0)
	    : QObject(p)
		, bDeleteAfterFirstUse(deleteAfterFirstUse)
		, qvData(data) {}

	static void curry(QObject *sender, const char *signal, QObject *receiver, const char *slot, QVariant data) {
		SignalCurry *c = new SignalCurry(data);
		connect(receiver, slot, c, SIGNAL(called(QVariant)));
		connect(sender, signal, c, SLOT(call));
	}

signals:
	void called(QVariant data);
public slots:
	void call() {
		emit called(qvData);
		if (bDeleteAfterFirstUse) deleteLater();
	}
};


#endif
