/* Copyright (c) 2007-2008 CSIRO
   Copyright (c) 2007-2009 Xiph.Org Foundation
   Copyright (c) 2008-2009 Gregory Maxwell 
   Written by Jean-Marc Valin and Gregory Maxwell */
/*
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   
   - Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
   - Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
   
   - Neither the name of the Xiph.org Foundation nor the names of its
   contributors may be used to endorse or promote products derived from
   this software without specific prior written permission.
   
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <math.h>
#include "bands.h"
#include "modes.h"
#include "vq.h"
#include "cwrs.h"
#include "stack_alloc.h"
#include "os_support.h"
#include "mathops.h"
#include "rate.h"

/* This is a cos() approximation designed to be bit-exact on any platform. Bit exactness
   with this approximation is important because it has an impact on the bit allocation */
static celt_int16 bitexact_cos(celt_int16 x)
{
   celt_int32 tmp;
   celt_int16 x2;
   tmp = (4096+((celt_int32)(x)*(x)))>>13;
   if (tmp > 32767)
      tmp = 32767;
   x2 = tmp;
   x2 = (32767-x2) + FRAC_MUL16(x2, (-7651 + FRAC_MUL16(x2, (8277 + FRAC_MUL16(-626, x2)))));
   if (x2 > 32766)
      x2 = 32766;
   return 1+x2;
}


#ifdef FIXED_POINT
/* Compute the amplitude (sqrt energy) in each of the bands */
void compute_band_energies(const CELTMode *m, const celt_sig *X, celt_ener *bank, int end, int _C, int M)
{
   int i, c, N;
   const celt_int16 *eBands = m->eBands;
   const int C = CHANNELS(_C);
   N = M*m->shortMdctSize;
   c=0; do {
      for (i=0;i<end;i++)
      {
         int j;
         celt_word32 maxval=0;
         celt_word32 sum = 0;
         
         j=M*eBands[i]; do {
            maxval = MAX32(maxval, X[j+c*N]);
            maxval = MAX32(maxval, -X[j+c*N]);
         } while (++j<M*eBands[i+1]);
         
         if (maxval > 0)
         {
            int shift = celt_ilog2(maxval)-10;
            j=M*eBands[i]; do {
               sum = MAC16_16(sum, EXTRACT16(VSHR32(X[j+c*N],shift)),
                                   EXTRACT16(VSHR32(X[j+c*N],shift)));
            } while (++j<M*eBands[i+1]);
            /* We're adding one here to make damn sure we never end up with a pitch vector that's
               larger than unity norm */
            bank[i+c*m->nbEBands] = EPSILON+VSHR32(EXTEND32(celt_sqrt(sum)),-shift);
         } else {
            bank[i+c*m->nbEBands] = EPSILON;
         }
         /*printf ("%f ", bank[i+c*m->nbEBands]);*/
      }
   } while (++c<C);
   /*printf ("\n");*/
}

/* Normalise each band such that the energy is one. */
void normalise_bands(const CELTMode *m, const celt_sig * restrict freq, celt_norm * restrict X, const celt_ener *bank, int end, int _C, int M)
{
   int i, c, N;
   const celt_int16 *eBands = m->eBands;
   const int C = CHANNELS(_C);
   N = M*m->shortMdctSize;
   c=0; do {
      i=0; do {
         celt_word16 g;
         int j,shift;
         celt_word16 E;
         shift = celt_zlog2(bank[i+c*m->nbEBands])-13;
         E = VSHR32(bank[i+c*m->nbEBands], shift);
         g = EXTRACT16(celt_rcp(SHL32(E,3)));
         j=M*eBands[i]; do {
            X[j+c*N] = MULT16_16_Q15(VSHR32(freq[j+c*N],shift-1),g);
         } while (++j<M*eBands[i+1]);
      } while (++i<end);
   } while (++c<C);
}

#else /* FIXED_POINT */
/* Compute the amplitude (sqrt energy) in each of the bands */
void compute_band_energies(const CELTMode *m, const celt_sig *X, celt_ener *bank, int end, int _C, int M)
{
   int i, c, N;
   const celt_int16 *eBands = m->eBands;
   const int C = CHANNELS(_C);
   N = M*m->shortMdctSize;
   c=0; do {
      for (i=0;i<end;i++)
      {
         int j;
         celt_word32 sum = 1e-10f;
         for (j=M*eBands[i];j<M*eBands[i+1];j++)
            sum += X[j+c*N]*X[j+c*N];
         bank[i+c*m->nbEBands] = celt_sqrt(sum);
         /*printf ("%f ", bank[i+c*m->nbEBands]);*/
      }
   } while (++c<C);
   /*printf ("\n");*/
}

/* Normalise each band such that the energy is one. */
void normalise_bands(const CELTMode *m, const celt_sig * restrict freq, celt_norm * restrict X, const celt_ener *bank, int end, int _C, int M)
{
   int i, c, N;
   const celt_int16 *eBands = m->eBands;
   const int C = CHANNELS(_C);
   N = M*m->shortMdctSize;
   c=0; do {
      for (i=0;i<end;i++)
      {
         int j;
         celt_word16 g = 1.f/(1e-10f+bank[i+c*m->nbEBands]);
         for (j=M*eBands[i];j<M*eBands[i+1];j++)
            X[j+c*N] = freq[j+c*N]*g;
      }
   } while (++c<C);
}

#endif /* FIXED_POINT */

/* De-normalise the energy to produce the synthesis from the unit-energy bands */
void denormalise_bands(const CELTMode *m, const celt_norm * restrict X, celt_sig * restrict freq, const celt_ener *bank, int end, int _C, int M)
{
   int i, c, N;
   const celt_int16 *eBands = m->eBands;
   const int C = CHANNELS(_C);
   N = M*m->shortMdctSize;
   celt_assert2(C<=2, "denormalise_bands() not implemented for >2 channels");
   c=0; do {
      celt_sig * restrict f;
      const celt_norm * restrict x;
      f = freq+c*N;
      x = X+c*N;
      for (i=0;i<end;i++)
      {
         int j, band_end;
         celt_word32 g = SHR32(bank[i+c*m->nbEBands],1);
         j=M*eBands[i];
         band_end = M*eBands[i+1];
         do {
            *f++ = SHL32(MULT16_32_Q15(*x, g),2);
            x++;
         } while (++j<band_end);
      }
      for (i=M*eBands[m->nbEBands];i<N;i++)
         *f++ = 0;
   } while (++c<C);
}

static void intensity_stereo(const CELTMode *m, celt_norm *X, celt_norm *Y, const celt_ener *bank, int bandID, int N)
{
   int i = bandID;
   int j;
   celt_word16 a1, a2;
   celt_word16 left, right;
   celt_word16 norm;
#ifdef FIXED_POINT
   int shift = celt_zlog2(MAX32(bank[i], bank[i+m->nbEBands]))-13;
#endif
   left = VSHR32(bank[i],shift);
   right = VSHR32(bank[i+m->nbEBands],shift);
   norm = EPSILON + celt_sqrt(EPSILON+MULT16_16(left,left)+MULT16_16(right,right));
   a1 = DIV32_16(SHL32(EXTEND32(left),14),norm);
   a2 = DIV32_16(SHL32(EXTEND32(right),14),norm);
   for (j=0;j<N;j++)
   {
      celt_norm r, l;
      l = X[j];
      r = Y[j];
      X[j] = MULT16_16_Q14(a1,l) + MULT16_16_Q14(a2,r);
      /* Side is not encoded, no need to calculate */
   }
}

static void stereo_split(celt_norm *X, celt_norm *Y, int N)
{
   int j;
   for (j=0;j<N;j++)
   {
      celt_norm r, l;
      l = MULT16_16_Q15(QCONST16(.70711f,15), X[j]);
      r = MULT16_16_Q15(QCONST16(.70711f,15), Y[j]);
      X[j] = l+r;
      Y[j] = r-l;
   }
}

static void stereo_merge(celt_norm *X, celt_norm *Y, celt_word16 mid, celt_word16 side, int N)
{
   int j;
   celt_word32 xp=0;
   celt_word32 El, Er;
#ifdef FIXED_POINT
   int kl, kr;
#endif
   celt_word32 t, lgain, rgain;

   /* Compute the norm of X+Y and X-Y as |X|^2 + |Y|^2 +/- sum(xy) */
   for (j=0;j<N;j++)
      xp = MAC16_16(xp, X[j], Y[j]);
   /* mid and side are in Q15, not Q14 like X and Y */
   mid = SHR32(mid, 1);
   side = SHR32(side, 1);
   El = MULT16_16(mid, mid) + MULT16_16(side, side) - 2*xp;
   Er = MULT16_16(mid, mid) + MULT16_16(side, side) + 2*xp;
   if (Er < EPSILON)
      Er = EPSILON;
   if (El < EPSILON)
      El = EPSILON;

#ifdef FIXED_POINT
   kl = celt_ilog2(El)>>1;
   kr = celt_ilog2(Er)>>1;
#endif
   t = VSHR32(El, (kl-7)<<1);
   lgain = celt_rsqrt_norm(t);
   t = VSHR32(Er, (kr-7)<<1);
   rgain = celt_rsqrt_norm(t);

#ifdef FIXED_POINT
   if (kl < 7)
      kl = 7;
   if (kr < 7)
      kr = 7;
#endif

   for (j=0;j<N;j++)
   {
      celt_norm r, l;
      l = X[j];
      r = Y[j];
      X[j] = EXTRACT16(PSHR32(MULT16_16(lgain, SUB16(l,r)), kl+1));
      Y[j] = EXTRACT16(PSHR32(MULT16_16(rgain, ADD16(l,r)), kr+1));
   }
}

/* Decide whether we should spread the pulses in the current frame */
int spreading_decision(const CELTMode *m, celt_norm *X, int *average, int last_decision, int end, int _C, int M)
{
   int i, c, N0;
   int sum = 0, nbBands=0;
   const int C = CHANNELS(_C);
   const celt_int16 * restrict eBands = m->eBands;
   int decision;
   
   N0 = M*m->shortMdctSize;

   if (M*(eBands[end]-eBands[end-1]) <= 8)
      return SPREAD_NONE;
   c=0; do {
      for (i=0;i<end;i++)
      {
         int j, N, tmp=0;
         int tcount[3] = {0};
         celt_norm * restrict x = X+M*eBands[i]+c*N0;
         N = M*(eBands[i+1]-eBands[i]);
         if (N<=8)
            continue;
         /* Compute rough CDF of |x[j]| */
         for (j=0;j<N;j++)
         {
            celt_word32 x2N; /* Q13 */

            x2N = MULT16_16(MULT16_16_Q15(x[j], x[j]), N);
            if (x2N < QCONST16(0.25f,13))
               tcount[0]++;
            if (x2N < QCONST16(0.0625f,13))
               tcount[1]++;
            if (x2N < QCONST16(0.015625f,13))
               tcount[2]++;
         }

         tmp = (2*tcount[2] >= N) + (2*tcount[1] >= N) + (2*tcount[0] >= N);
         sum += tmp*256;
         nbBands++;
      }
   } while (++c<C);
   sum /= nbBands;
   /* Recursive averaging */
   sum = (sum+*average)>>1;
   *average = sum;
   /* Hysteresis */
   sum = (3*sum + (((3-last_decision)<<7) + 64) + 2)>>2;
   if (sum < 80)
   {
      decision = SPREAD_AGGRESSIVE;
   } else if (sum < 256)
   {
      decision = SPREAD_NORMAL;
   } else if (sum < 384)
   {
      decision = SPREAD_LIGHT;
   } else {
      decision = SPREAD_NONE;
   }
   return decision;
}

#ifdef MEASURE_NORM_MSE

float MSE[30] = {0};
int nbMSEBands = 0;
int MSECount[30] = {0};

void dump_norm_mse(void)
{
   int i;
   for (i=0;i<nbMSEBands;i++)
   {
      printf ("%g ", MSE[i]/MSECount[i]);
   }
   printf ("\n");
}

void measure_norm_mse(const CELTMode *m, float *X, float *X0, float *bandE, float *bandE0, int M, int N, int C)
{
   static int init = 0;
   int i;
   if (!init)
   {
      atexit(dump_norm_mse);
      init = 1;
   }
   for (i=0;i<m->nbEBands;i++)
   {
      int j;
      int c;
      float g;
      if (bandE0[i]<10 || (C==2 && bandE0[i+m->nbEBands]<1))
         continue;
      c=0; do {
         g = bandE[i+c*m->nbEBands]/(1e-15+bandE0[i+c*m->nbEBands]);
         for (j=M*m->eBands[i];j<M*m->eBands[i+1];j++)
            MSE[i] += (g*X[j+c*N]-X0[j+c*N])*(g*X[j+c*N]-X0[j+c*N]);
      } while (++c<C);
      MSECount[i]+=C;
   }
   nbMSEBands = m->nbEBands;
}

#endif

static void interleave_vector(celt_norm *X, int N0, int stride)
{
   int i,j;
   VARDECL(celt_norm, tmp);
   int N;
   SAVE_STACK;
   N = N0*stride;
   ALLOC(tmp, N, celt_norm);
   for (i=0;i<stride;i++)
      for (j=0;j<N0;j++)
         tmp[j*stride+i] = X[i*N0+j];
   for (j=0;j<N;j++)
      X[j] = tmp[j];
   RESTORE_STACK;
}

static void deinterleave_vector(celt_norm *X, int N0, int stride)
{
   int i,j;
   VARDECL(celt_norm, tmp);
   int N;
   SAVE_STACK;
   N = N0*stride;
   ALLOC(tmp, N, celt_norm);
   for (i=0;i<stride;i++)
      for (j=0;j<N0;j++)
         tmp[i*N0+j] = X[j*stride+i];
   for (j=0;j<N;j++)
      X[j] = tmp[j];
   RESTORE_STACK;
}

void haar1(celt_norm *X, int N0, int stride)
{
   int i, j;
   N0 >>= 1;
   for (i=0;i<stride;i++)
      for (j=0;j<N0;j++)
      {
         celt_norm tmp1, tmp2;
         tmp1 = MULT16_16_Q15(QCONST16(.7070678f,15), X[stride*2*j+i]);
         tmp2 = MULT16_16_Q15(QCONST16(.7070678f,15), X[stride*(2*j+1)+i]);
         X[stride*2*j+i] = tmp1 + tmp2;
         X[stride*(2*j+1)+i] = tmp1 - tmp2;
      }
}

static int compute_qn(int N, int b, int offset, int stereo)
{
   static const celt_int16 exp2_table8[8] =
      {16384, 17866, 19483, 21247, 23170, 25267, 27554, 30048};
   int qn, qb;
   int N2 = 2*N-1;
   if (stereo && N==2)
      N2--;
   qb = (b+N2*offset)/N2;
   if (qb > (b>>1)-(1<<BITRES))
      qb = (b>>1)-(1<<BITRES);

   if (qb<0)
       qb = 0;
   if (qb>8<<BITRES)
     qb = 8<<BITRES;

   if (qb<(1<<BITRES>>1)) {
      qn = 1;
   } else {
      qn = exp2_table8[qb&0x7]>>(14-(qb>>BITRES));
      qn = (qn+1)>>1<<1;
   }
   celt_assert(qn <= 256);
   return qn;
}

static celt_uint32 lcg_rand(celt_uint32 seed)
{
   return 1664525 * seed + 1013904223;
}

/* This function is responsible for encoding and decoding a band for both
   the mono and stereo case. Even in the mono case, it can split the band
   in two and transmit the energy difference with the two half-bands. It
   can be called recursively so bands can end up being split in 8 parts. */
static void quant_band(int encode, const CELTMode *m, int i, celt_norm *X, celt_norm *Y,
      int N, int b, int spread, int B, int intensity, int tf_change, celt_norm *lowband, int resynth, void *ec,
      celt_int32 *remaining_bits, int LM, celt_norm *lowband_out, const celt_ener *bandE, int level,
      celt_int32 *seed, celt_word16 gain, celt_norm *lowband_scratch, int fill)
{
   int q;
   int curr_bits;
   int stereo, split;
   int imid=0, iside=0;
   int N0=N;
   int N_B=N;
   int N_B0;
   int B0=B;
   int time_divide=0;
   int recombine=0;
   int inv = 0;
   celt_word16 mid=0, side=0;

   N_B /= B;
   N_B0 = N_B;

   split = stereo = Y != NULL;

   /* Special case for one sample */
   if (N==1)
   {
      int c;
      celt_norm *x = X;
      c=0; do {
         int sign=0;
         if (*remaining_bits>=1<<BITRES)
         {
            if (encode)
            {
               sign = x[0]<0;
               ec_enc_bits((ec_enc*)ec, sign, 1);
            } else {
               sign = ec_dec_bits((ec_dec*)ec, 1);
            }
            *remaining_bits -= 1<<BITRES;
            b-=1<<BITRES;
         }
         if (resynth)
            x[0] = sign ? -NORM_SCALING : NORM_SCALING;
         x = Y;
      } while (++c<1+stereo);
      if (lowband_out)
         lowband_out[0] = SHR16(X[0],4);
      return;
   }

   if (!stereo && level == 0)
   {
      int k;
      if (tf_change>0)
         recombine = tf_change;
      /* Band recombining to increase frequency resolution */

      if (lowband && (recombine || ((N_B&1) == 0 && tf_change<0) || B0>1))
      {
         int j;
         for (j=0;j<N;j++)
            lowband_scratch[j] = lowband[j];
         lowband = lowband_scratch;
      }

      for (k=0;k<recombine;k++)
      {
         B>>=1;
         N_B<<=1;
         if (encode)
            haar1(X, N_B, B);
         if (lowband)
            haar1(lowband, N_B, B);
      }

      /* Increasing the time resolution */
      while ((N_B&1) == 0 && tf_change<0)
      {
         if (encode)
            haar1(X, N_B, B);
         if (lowband)
            haar1(lowband, N_B, B);
         B <<= 1;
         N_B >>= 1;
         time_divide++;
         tf_change++;
      }
      B0=B;
      N_B0 = N_B;

      /* Reorganize the samples in time order instead of frequency order */
      if (B0>1)
      {
         if (encode)
            deinterleave_vector(X, N_B, B0);
         if (lowband)
            deinterleave_vector(lowband, N_B, B0);
      }
   }

   /* If we need more than 32 bits, try splitting the band in two. */
   if (!stereo && LM != -1 && b > 32<<BITRES && N>2)
   {
      if (LM>0 || (N&1)==0)
      {
         N >>= 1;
         Y = X+N;
         split = 1;
         LM -= 1;
         B = (B+1)>>1;
      }
   }

   if (split)
   {
      int qn;
      int itheta=0;
      int mbits, sbits, delta;
      int qalloc;
      int offset;

      /* Decide on the resolution to give to the split parameter theta */
      offset = ((m->logN[i]+(LM<<BITRES))>>1) - (stereo ? QTHETA_OFFSET_STEREO : QTHETA_OFFSET);
      qn = compute_qn(N, b, offset, stereo);
      qalloc = 0;
      if (stereo && i>=intensity)
         qn = 1;
      if (encode)
      {
         /* theta is the atan() of the ratio between the (normalized)
            side and mid. With just that parameter, we can re-scale both
            mid and side because we know that 1) they have unit norm and
            2) they are orthogonal. */
         itheta = stereo_itheta(X, Y, stereo, N);
      }
      if (qn!=1)
      {
         if (encode)
            itheta = (itheta*qn+8192)>>14;

         /* Entropy coding of the angle. We use a uniform pdf for the
            first stereo split but a triangular one for the rest. */
         if (stereo || B>1)
         {
            if (encode)
               ec_enc_uint((ec_enc*)ec, itheta, qn+1);
            else
               itheta = ec_dec_uint((ec_dec*)ec, qn+1);
            qalloc = log2_frac(qn+1,BITRES);
         } else {
            int fs=1, ft;
            ft = ((qn>>1)+1)*((qn>>1)+1);
            if (encode)
            {
               int fl;

               fs = itheta <= (qn>>1) ? itheta + 1 : qn + 1 - itheta;
               fl = itheta <= (qn>>1) ? itheta*(itheta + 1)>>1 :
                ft - ((qn + 1 - itheta)*(qn + 2 - itheta)>>1);

               ec_encode((ec_enc*)ec, fl, fl+fs, ft);
            } else {
               int fl=0;
               int fm;
               fm = ec_decode((ec_dec*)ec, ft);

               if (fm < ((qn>>1)*((qn>>1) + 1)>>1))
               {
                  itheta = (isqrt32(8*(celt_uint32)fm + 1) - 1)>>1;
                  fs = itheta + 1;
                  fl = itheta*(itheta + 1)>>1;
               }
               else
               {
                  itheta = (2*(qn + 1)
                   - isqrt32(8*(celt_uint32)(ft - fm - 1) + 1))>>1;
                  fs = qn + 1 - itheta;
                  fl = ft - ((qn + 1 - itheta)*(qn + 2 - itheta)>>1);
               }

               ec_dec_update((ec_dec*)ec, fl, fl+fs, ft);
            }
            qalloc = log2_frac(ft,BITRES) - log2_frac(fs,BITRES) + 1;
         }
         itheta = (celt_int32)itheta*16384/qn;
         if (encode && stereo)
            stereo_split(X, Y, N);
         /* TODO: Renormalising X and Y *may* help fixed-point a bit at very high rate.
                  Let's do that at higher complexity */
      } else if (stereo) {
         if (encode)
         {
            inv = itheta > 8192;
            if (inv)
            {
               int j;
               for (j=0;j<N;j++)
                  Y[j] = -Y[j];
            }
            intensity_stereo(m, X, Y, bandE, i, N);
         }
         if (b>2<<BITRES && *remaining_bits > 2<<BITRES)
         {
            if (encode)
               ec_enc_bit_logp(ec, inv, 2);
            else
               inv = ec_dec_bit_logp(ec, 2);
            qalloc = inv ? 16 : 4;
         } else
            inv = 0;
         itheta = 0;
      }

      if (itheta == 0)
      {
         imid = 32767;
         iside = 0;
         delta = -10000;
      } else if (itheta == 16384)
      {
         imid = 0;
         iside = 32767;
         delta = 10000;
      } else {
         imid = bitexact_cos(itheta);
         iside = bitexact_cos(16384-itheta);
         /* This is the mid vs side allocation that minimizes squared error
            in that band. */
         delta = (N-1)*(log2_frac(iside,BITRES+2)-log2_frac(imid,BITRES+2))>>2;
      }

#ifdef FIXED_POINT
      mid = imid;
      side = iside;
#else
      mid = (1.f/32768)*imid;
      side = (1.f/32768)*iside;
#endif

      /* This is a special case for N=2 that only works for stereo and takes
         advantage of the fact that mid and side are orthogonal to encode
         the side with just one bit. */
      if (N==2 && stereo)
      {
         int c;
         int sign=1;
         celt_norm *x2, *y2;
         mbits = b-qalloc;
         sbits = 0;
         /* Only need one bit for the side */
         if (itheta != 0 && itheta != 16384)
            sbits = 1<<BITRES;
         mbits -= sbits;
         c = itheta > 8192;
         *remaining_bits -= qalloc+sbits;

         x2 = c ? Y : X;
         y2 = c ? X : Y;
         if (sbits)
         {
            if (encode)
            {
               /* Here we only need to encode a sign for the side */
               sign = x2[0]*y2[1] - x2[1]*y2[0] > 0;
               ec_enc_bits((ec_enc*)ec, sign, 1);
            } else {
               sign = ec_dec_bits((ec_dec*)ec, 1);
            }
         }
         sign = 2*sign - 1;
         quant_band(encode, m, i, x2, NULL, N, mbits, spread, B, intensity, tf_change, lowband, resynth, ec, remaining_bits, LM, lowband_out, NULL, level, seed, gain, lowband_scratch, fill);
         y2[0] = -sign*x2[1];
         y2[1] = sign*x2[0];
         if (resynth)
         {
            celt_norm tmp;
            X[0] = MULT16_16_Q15(mid, X[0]);
            X[1] = MULT16_16_Q15(mid, X[1]);
            Y[0] = MULT16_16_Q15(side, Y[0]);
            Y[1] = MULT16_16_Q15(side, Y[1]);
            tmp = X[0];
            X[0] = SUB16(tmp,Y[0]);
            Y[0] = ADD16(tmp,Y[0]);
            tmp = X[1];
            X[1] = SUB16(tmp,Y[1]);
            Y[1] = ADD16(tmp,Y[1]);
         }
      } else {
         /* "Normal" split code */
         celt_norm *next_lowband2=NULL;
         celt_norm *next_lowband_out1=NULL;
         int next_level=0;

         /* Give more bits to low-energy MDCTs than they would otherwise deserve */
         if (B>1 && !stereo && itheta > 8192)
            delta -= delta>>(1+level);

         mbits = (b-qalloc-delta)/2;
         if (mbits > b-qalloc)
            mbits = b-qalloc;
         if (mbits<0)
            mbits=0;
         sbits = b-qalloc-mbits;
         *remaining_bits -= qalloc;

         if (lowband && !stereo)
            next_lowband2 = lowband+N; /* >32-bit split case */

         /* Only stereo needs to pass on lowband_out. Otherwise, it's
            handled at the end */
         if (stereo)
            next_lowband_out1 = lowband_out;
         else
            next_level = level+1;

         quant_band(encode, m, i, X, NULL, N, mbits, spread, B, intensity, tf_change,
               lowband, resynth, ec, remaining_bits, LM, next_lowband_out1,
               NULL, next_level, seed, MULT16_16_P15(gain,mid), lowband_scratch, fill);
         quant_band(encode, m, i, Y, NULL, N, sbits, spread, B, intensity, tf_change,
               next_lowband2, resynth, ec, remaining_bits, LM, NULL,
               NULL, next_level, seed, MULT16_16_P15(gain,side), NULL, fill && !stereo);
      }

   } else {
      /* This is the basic no-split case */
      q = bits2pulses(m, i, LM, b);
      curr_bits = pulses2bits(m, i, LM, q);
      *remaining_bits -= curr_bits;

      /* Ensures we can never bust the budget */
      while (*remaining_bits < 0 && q > 0)
      {
         *remaining_bits += curr_bits;
         q--;
         curr_bits = pulses2bits(m, i, LM, q);
         *remaining_bits -= curr_bits;
      }

      if (q!=0)
      {
         int K = get_pulses(q);

         /* Finally do the actual quantization */
         if (encode)
            alg_quant(X, N, K, spread, B, lowband, resynth, (ec_enc*)ec, seed, gain);
         else
            alg_unquant(X, N, K, spread, B, lowband, (ec_dec*)ec, seed, gain);
      } else {
         /* If there's no pulse, fill the band anyway */
         int j;
         if (resynth)
         {
            if (!fill)
            {
               for (j=0;j<N;j++)
                  X[j] = 0;
            } else {
               if (lowband == NULL || (spread==SPREAD_AGGRESSIVE && B<=1))
               {
                  /* Noise */
                  for (j=0;j<N;j++)
                  {
                     *seed = lcg_rand(*seed);
                     X[j] = (int)(*seed)>>20;
                  }
               } else {
                  /* Folded spectrum */
                  for (j=0;j<N;j++)
                     X[j] = lowband[j];
               }
               renormalise_vector(X, N, gain);
            }
         }
      }
   }

   /* This code is used by the decoder and by the resynthesis-enabled encoder */
   if (resynth)
   {
      if (stereo)
      {
         if (N!=2)
            stereo_merge(X, Y, mid, side, N);
         if (inv)
         {
            int j;
            for (j=0;j<N;j++)
               Y[j] = -Y[j];
         }
      } else if (level == 0)
      {
         int k;

         /* Undo the sample reorganization going from time order to frequency order */
         if (B0>1)
            interleave_vector(X, N_B, B0);

         /* Undo time-freq changes that we did earlier */
         N_B = N_B0;
         B = B0;
         for (k=0;k<time_divide;k++)
         {
            B >>= 1;
            N_B <<= 1;
            haar1(X, N_B, B);
         }

         for (k=0;k<recombine;k++)
         {
            haar1(X, N_B, B);
            N_B>>=1;
            B <<= 1;
         }

         /* Scale output for later folding */
         if (lowband_out)
         {
            int j;
            celt_word16 n;
            n = celt_sqrt(SHL32(EXTEND32(N0),22));
            for (j=0;j<N0;j++)
               lowband_out[j] = MULT16_16_Q15(n,X[j]);
         }
      }
   }
}

void quant_all_bands(int encode, const CELTMode *m, int start, int end,
      celt_norm *_X, celt_norm *_Y, const celt_ener *bandE, int *pulses,
      int shortBlocks, int spread, int dual_stereo, int intensity, int *tf_res, int resynth,
      int total_bits, void *ec, int LM, int codedBands)
{
   int i, balance;
   celt_int32 remaining_bits;
   const celt_int16 * restrict eBands = m->eBands;
   celt_norm * restrict norm, * restrict norm2;
   VARDECL(celt_norm, _norm);
   VARDECL(celt_norm, lowband_scratch);
   int B;
   int M;
   celt_int32 seed;
   int lowband_offset;
   int update_lowband = 1;
   int C = _Y != NULL ? 2 : 1;
   SAVE_STACK;

   M = 1<<LM;
   B = shortBlocks ? M : 1;
   ALLOC(_norm, C*M*eBands[m->nbEBands], celt_norm);
   ALLOC(lowband_scratch, M*(eBands[m->nbEBands]-eBands[m->nbEBands-1]), celt_norm);
   norm = _norm;
   norm2 = norm + M*eBands[m->nbEBands];
#if 0
   if (C==2)
   {
      int j;
      int left = 0;
      for (j=intensity;j<codedBands;j++)
      {
         int tmp = pulses[j]/2;
         left += tmp;
         pulses[j] -= tmp;
      }
      if (codedBands) {
         int perband;
         perband = left/(m->eBands[codedBands]-m->eBands[start]);
         for (j=start;j<codedBands;j++)
            pulses[j] += perband*(m->eBands[j+1]-m->eBands[j]);
         left = left-(m->eBands[codedBands]-m->eBands[start])*perband;
         for (j=start;j<codedBands;j++)
         {
            int tmp = IMIN(left, m->eBands[j+1]-m->eBands[j]);
            pulses[j] += tmp;
            left -= tmp;
         }
      }
   }
#endif
   if (encode)
      seed = ((ec_enc*)ec)->rng;
   else
      seed = ((ec_dec*)ec)->rng;
   balance = 0;
   lowband_offset = -1;
   for (i=start;i<end;i++)
   {
      int tell;
      int b;
      int N;
      int curr_balance;
      int effective_lowband=-1;
      celt_norm * restrict X, * restrict Y;
      int tf_change=0;
      
      X = _X+M*eBands[i];
      if (_Y!=NULL)
         Y = _Y+M*eBands[i];
      else
         Y = NULL;
      N = M*eBands[i+1]-M*eBands[i];
      if (encode)
         tell = ec_enc_tell((ec_enc*)ec, BITRES);
      else
         tell = ec_dec_tell((ec_dec*)ec, BITRES);

      /* Compute how many bits we want to allocate to this band */
      if (i != start)
         balance -= tell;
      remaining_bits = (total_bits<<BITRES)-tell-1;
      if (i <= codedBands-1)
      {
         curr_balance = (codedBands-i);
         if (curr_balance > 3)
            curr_balance = 3;
         curr_balance = balance / curr_balance;
         b = IMIN(remaining_bits+1,pulses[i]+curr_balance);
         if (b<0)
            b = 0;
      } else {
         b = 0;
      }
      /* Prevents ridiculous bit depths */
      if (b > C*16*N<<BITRES)
         b = C*16*N<<BITRES;

      if (M*eBands[i]-N >= M*eBands[start] && (update_lowband || lowband_offset==-1))
            lowband_offset = M*eBands[i];

      tf_change = tf_res[i];
      if (i>=m->effEBands)
      {
         X=norm;
         if (_Y!=NULL)
            Y = norm;
      }

      /* This ensures we never repeat spectral content within one band */
      if (lowband_offset != -1)
      {
         effective_lowband = lowband_offset-N;
         if (effective_lowband < M*eBands[start])
            effective_lowband = M*eBands[start];
      }
      if (dual_stereo && i==intensity)
      {
         int j;

         /* Switch off dual stereo to do intensity */
         dual_stereo = 0;
         for (j=M*eBands[start];j<M*eBands[i];j++)
            norm[j] = HALF32(norm[j]+norm2[j]);
      }
      if (dual_stereo)
      {
         quant_band(encode, m, i, X, NULL, N, b/2, spread, B, intensity, tf_change,
               effective_lowband != -1 ? norm+effective_lowband : NULL, resynth, ec, &remaining_bits, LM,
               norm+M*eBands[i], bandE, 0, &seed, Q15ONE, lowband_scratch, 1);
         quant_band(encode, m, i, Y, NULL, N, b/2, spread, B, intensity, tf_change,
               effective_lowband != -1 ? norm2+effective_lowband : NULL, resynth, ec, &remaining_bits, LM,
               norm2+M*eBands[i], bandE, 0, &seed, Q15ONE, lowband_scratch, 1);
      } else {
         quant_band(encode, m, i, X, Y, N, b, spread, B, intensity, tf_change,
               effective_lowband != -1 ? norm+effective_lowband : NULL, resynth, ec, &remaining_bits, LM,
               norm+M*eBands[i], bandE, 0, &seed, Q15ONE, lowband_scratch, 1);
      }
      balance += pulses[i] + tell;

      /* Update the folding position only as long as we have 1 bit/sample depth */
      update_lowband = (b>>BITRES)>N;
   }
   RESTORE_STACK;
}

