/* Copyright (C) 2005-2009, Thorvald Natvig <thorvald@natvig.com>

   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   - Redistributions of source code must retain the above copyright notice,
     this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright notice,
     this list of conditions and the following disclaimer in the documentation
     and/or other materials provided with the distribution.
   - Neither the name of the Mumble Developers nor the names of its
     contributors may be used to endorse or promote products derived from this
     software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef _SETTINGS_H
#define _SETTINGS_H

#include "mumble_pch.hpp"

// Global helper classes to spread variables around across threads
// especially helpfull to initialize things like the stored
// preference for audio transmission, since the GUI elements
// will be created long before the AudioInput object, and the
// latter lives in a separate thread and so cannot touch the
// GUI.

struct Shortcut {
	int iIndex;
	QList<QVariant> qlButtons;
	QVariant qvData;
	bool bSuppress;
	bool operator <(const Shortcut &) const;
	bool isServerSpecific() const;
	bool operator ==(const Shortcut &) const;
};

struct ShortcutTarget {
	bool bUsers;
	QStringList qlUsers;
	QList<unsigned int> qlSessions;
	int iChannel;
	QString qsGroup;
	bool bLinks;
	bool bChildren;
	bool bForceCenter;
	ShortcutTarget();
	bool isServerSpecific() const;
	bool operator <(const ShortcutTarget &) const;
	bool operator ==(const ShortcutTarget &) const;
};

quint32 qHash(const ShortcutTarget &);
quint32 qHash(const QList<ShortcutTarget> &);

QDataStream &operator<<(QDataStream &, const ShortcutTarget &);
QDataStream &operator>>(QDataStream &, ShortcutTarget &);
Q_DECLARE_METATYPE(ShortcutTarget);

struct Settings {
	enum AudioTransmit { Continous, VAD, PushToTalk };
	enum VADSource { Amplitude, SignalToNoise };
	enum LoopMode { None, Local, Server };
	enum OverlayShow { Nothing, Talking, All };
	enum ChannelExpand { NoChannels, ChannelsWithUsers, AllChannels };
	enum ChannelDrag { Ask, DoNothing, Move };
	enum ServerShow { ShowFavorite, ShowPopulated, ShowReachable, ShowAll };
	typedef QPair<QList<QSslCertificate>, QSslKey> KeyPair;

	AudioTransmit atTransmit;
	quint64 uiDoublePush;

	bool bExpert;

	bool bPushClick;
	static const QString cqsDefaultPushClickOn;
	static const QString cqsDefaultPushClickOff;
	QString qsPushClickOn;
	QString qsPushClickOff;

	bool bTransmitPosition;
	bool bMute, bDeaf;
	bool bTTS;
	bool bUserTop;
	bool bWhisperFriends;
	int iTTSVolume, iTTSThreshold;
	int iQuality, iMinLoudness, iVoiceHold, iJitterBufferSize;
	int iNoiseSuppress;
	unsigned int iIdleTime;
	VADSource vsVAD;
	float fVADmin, fVADmax;
	int iFramesPerPacket;
	QString qsAudioInput, qsAudioOutput;
	float fVolume;
	float fOtherVolume;
	int iOutputDelay;

	QString qsALSAInput, qsALSAOutput;
	QString qsPulseAudioInput, qsPulseAudioOutput;
	QString qsOSSInput, qsOSSOutput;
	int iPortAudioInput, iPortAudioOutput;
	QString qsASIOclass;
	QList<QVariant> qlASIOmic;
	QList<QVariant> qlASIOspeaker;

	QString qsWASAPIInput, qsWASAPIOutput;

	QByteArray qbaDXInput, qbaDXOutput;

	bool bEcho;
	bool bEchoMulti;
	bool bPositionalAudio;
	bool bPositionalHeadphone;
	float fAudioMinDistance, fAudioMaxDistance, fAudioMaxDistVolume, fAudioBloom;

	bool bOverlayEnable;
	bool bOverlayUserTextures;
	OverlayShow osOverlay;
	bool bOverlayAlwaysSelf;
	float fOverlayX;
	float fOverlayY;
	bool bOverlayLeft, bOverlayRight, bOverlayTop, bOverlayBottom;
	QFont qfOverlayFont;
	float fOverlayHeight;
	QColor qcOverlayUser;
	QColor qcOverlayTalking;
	QColor qcOverlayWhisper;
	QColor qcOverlayChannel;
	QColor qcOverlayChannelTalking;

	int iLCDUserViewMinColWidth;
	int iLCDUserViewSplitterWidth;
	QMap<QString, bool> qmLCDDevices;

	QList<Shortcut> qlShortcuts;

	enum MessageLog { LogConsole = 0x01, LogTTS = 0x02, LogBalloon = 0x04, LogSoundfile = 0x08};
	QMap<int, QString> qmMessageSounds;
	QMap<int, quint32> qmMessages;

	QString qsLanguage;
	QString qsStyle;
	QString qsSkin;
	bool bFirstTime;
	QByteArray qbaMainWindowGeometry, qbaMainWindowState, qbaMinimalViewGeometry, qbaSplitterState, qbaHeaderState;
	QByteArray qbaConfigGeometry;
	enum WindowLayout { LayoutClassic, LayoutStacked, LayoutHybrid, LayoutCustom };
	int iWindowLayout;
	ChannelExpand ceExpand;
	ChannelDrag ceChannelDrag;
	bool bShowChatbar;
	bool bMinimalView;
	bool bHideFrame;
	bool bAlwaysOnTop;
	bool bAskOnQuit;
	bool bHideTray;
	bool bUsage;
	bool bAdvancedACLCfg;

	QString qsUsername;
	QString qsLastServer;
	ServerShow ssFilter;

	bool bUpdateCheck;
	bool bPluginCheck;

	// Network settings
	enum ProxyType { NoProxy, HttpProxy, Socks5Proxy };
	bool bTCPCompat, bReconnect;
	ProxyType ptProxyType;
	QString qsProxyHost, qsProxyUsername, qsProxyPassword;
	unsigned short usProxyPort;

	static const int ciDefaultMaxImageSize = 50 * 1024; // Restrict to 50KiB as a default
	int iMaxImageSize;
	int iMaxImageWidth;
	int iMaxImageHeight;
	KeyPair kpCertificate;
	bool bSuppressIdentity;

	// Nonsaved
	LoopMode lmLoopMode;
	float dPacketLoss;
	float dMaxPacketDelay;

	bool doEcho() const;
	bool doPositionalAudio() const;

	Settings();
	void load();
	void save();
};

#else
struct Settings;
#endif
